' SAP1 Computer, after Ben Eater and A.P. Malvino
' Rev 1.0.0 William M Leue 30-Dec-2021

option default integer
option base 0

' Constants

' Screen layout params
const MWIDTH = 340
const MHEIGHT = 100
const BWIDTH = MM.HRES - 2*MWIDTH
const DD_WIDTH = 25
const DD_HEIGHT = 40
const DD_NUM = 4

' Keyboard codes
const ESC   = 27
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const F1    = 145
const F2    = 146
const F3    = 147
const F4    = 148
const F5    = 149
const F8    = 152
const F12   = 156

' Computer module indices
const NMODULES = 12
const CLOCK   = 0
const PC      = 1
const ADDREG  = 2
const INSREG  = 3
const RAMM    = 4
const PROGRAM = 5
const AREG    = 6
const ALU     = 7
const BREG    = 8
const OUT     = 9
const FLAGS   = 10
const CNTRL   = 11

' Microcode ROM sizes and indices
const NBITS  = 8
const RSIZE  = 16
const NMICROSTEPS = 5
const NCONTROL_SIGS = 16
const NOPS = 16
const NOP_STEPS = 22

' conditional jump opcodes and flag bits
const CARRY_BIT  = &b00000001
const CARRY_MASK = &b11111110
const ZERO_BIT   = &b00000010
const ZERO_MASK  = &b11111101
const NEG_BIT    = &b10000000
const NEG_MASK   = &b01111111

' clock state
const CLK_STOPPED = 0
const CLK_RUNNING = 1
const CLK_SLIDER_INCREMENT = 15

' Control line bit positions
const HLT = 0
const RO  = 1
const IO  = 2
const CO  = 3
const CE  = 4
const AO  = 5
const EO  = 6
const FI  = 7
const SU  = 8
const MI  = 9
const RI  = 10
const II  = 11
const JMP = 12
const AI  = 13
const BI  = 14
const OI  = 15

' Opcode values (4 MSB of instruction register)
const opNOP = &b0000
const opLDA = &b0001
const opADD = &b0010
const opSUB = &b0011
const opSTA = &b0100
const opLDI = &b0101
const opJMP = &b0110
const opJC  = &b0111
const opJZ  = &b1000
const opJN  = &b1001
const opOUT = &b1110
const opHLT = &b1111

' display light params
const LTRAD = 5
const LTSPAC = 2*LTRAD+2
const NLIGHTS = 53
const NCOLORS = 4

' Digital output modes
const dHEXRAW = 1
const dUNSIGNED = 2
const dSIGNED = 3

' Simulated 7-segment display params
const CSIZE = 30
const NSEGS = 7
const NFIGS = 18
const SDISW = CSIZE-3
const SDISH = 2*SDISW-7
const SEGW = 4
const SS = 2
const MINUS = 16
const BLANK = 17
const XBAR = 3

const CHECK_DATA_ROOT = 123000

const MAX_PROGRAMS = 30

' mouse channel, if any
const MCHAN = 2

' Trace Constants
const MAX_TRACE_ROWS = 36
const MAX_TRACES = MAX_TRACE_ROWS*2

' Globals

' Screen location, color, and state info
dim mheights(NMODULES)
dim mnames$(NMODULES)
dim op_values(NOPS)
dim op_names$(NOPS)
dim light_data(5, NMODULES)
dim light_locs(2, NMODULES)
dim ctrlsig_names$(NCONTROL_SIGS)
dim ctrl_light_locs(2, NCONTROL_SIGS)
dim pctlocs(3, 3)
dim digital_display_locs(4, 3)
dim digit_mode = dSIGNED
dim colors(2, NCOLORS)
dim clk_ctrls(4, 5)
dim clk_rate = 500
dim clk_slider = 150
dim prv_clk_slider = clk_slider
dim digits_x = 0
dim digits_y = 0
dim clk_count = -1
dim pmode = 1
dim pctmoduleloc(2)

' microcode ROM info
dim mcr_fetch(NCONTROL_SIGS+2, 2)
dim mcr_ops(NCONTROL_SIGS+2, NOP_STEPS)
dim mcr_index(2, NOPS)

' Core computer registers and RAM
dim clk   = 0
dim pcr   = 0
dim insr  = 0
dim mr    = 0
dim argc  = 0
dim alur  = 0
dim alu_carry_out = 0
dim ar_zero = 0
dim ar_neg  = 0
dim br    = 0
dim outr  = 0
dim flagr = 0
dim bus   = 0
dim ram(RSIZE)
dim rs_ram(RSIZE)

' Core control lines and related params
dim cntrl_lines(NCONTROL_SIGS)
dim ustep = 0
dim opcode = 0
dim opstart = 0
dim opstep = 0
dim nopsteps = 0
dim opaddr = 0
dim op_nsteps = 0

' User control state
dim clk_state = CLK_STOPPED
dim clk_speed = 20

' 7-segment display character ROM
dim SegCoords(NSEGS, 6, 2)
dim SegActivations(NFIGS, NSEGS)

' Program Load and User Controls
dim pselect = 1
dim nprograms = 0
dim program_names$(MAX_PROGRAMS)
dim hasMouse = 0
dim loading_mode = 0
dim selected_program$
dim status_msg$ = ""

' trace regster
dim tr(5, MAX_TRACES)
dim trace_enable = 0
dim num_trace_entries = 0
dim trace_address = 0

' Main Program
open "debug.txt" for output as #1
ReadModuleInfo
CheckData 1
ReadClockControlLocations
CheckData 2
ReadMicrocodeRom
CheckData 3
ReadSegCoords
ReadSegActivations
CheckData 4
ReadDigitalDisplayLocations
CheckData 5
ReadPgrmCtrlLocations
CheckData 6
InitMouse
ResetComputer
DrawComputer
HandleKeyboardEvents
end

' Quick way of checking that all the data statements have been
' read correctly or at least fully.
sub CheckData num
  local val
  read val
  if val <> CHECK_DATA_ROOT + num then
    print "Incorrect Data at location ";num;"
    print "Expected ";CHECK_DATA_ROOT+num;", Read ";val
    end
  end if
end sub 

' Read data for modules:
'  Module heights on screen
'  Module names
'  Module display lights
'  Control line names
'  Light colors
sub ReadModuleInfo
  local i
  local float lpx, lpy, nl, lc1, lc2
  for i = 0 to NMODULES-1
    read mheights(i)
  next i
  for i = 0 to NMODULES-1
    read mnames$(i)
  next i
  for i = 0 to NMODULES-1
    read lpx, lpy, light_data(2, i), light_data(3, i), light_data(4, i)
    light_data(0, i) = int(lpx*MWIDTH + 0.5)
    light_data(1, i) = int(lpy*mheights(i))
  next i
  for i = 0 to NCONTROL_SIGS-1
    read ctrlsig_names$(i)
  next i
  for i = 0 to NCOLORS-1
    read colors(0, i) : read colors(1, i)
  next i
end sub

' Read the clock control widget locations
sub ReadClockControlLocations
  local i, j
  for i = 0 to 4
    for j = 0 to 3
      read clk_ctrls(j, i)
    next j
  next i
end sub

' Read the Microcode ROMs and index array, opcode info
sub ReadMicrocodeRom
  local i, j
  for i = 0 to 1
    for j = 0 to NCONTROL_SIGS+1
      read mcr_fetch(j, i)
    next j
  next i
  for i = 0 to NOP_STEPS-1
    for j = 0 to NCONTROL_SIGS+1
      read mcr_ops(j, i)
    next j
  next i
  for i = 0 to NOPS-1
    for j = 0 to 1
      read mcr_index(j, i)
    next j
  next i
  for i = 0 to NOPS-1
    read op_names$(i)
  next i
  for i = 0 to NOPS-1
    read op_values(i) 
  next i
end sub

' Read and Save locations for Digital Output Module
sub ReadDigitalDisplayLocations
  local i, j
  for i = 0 to 2
    for j = 0 to 3
      read digital_display_locs(j, i)
    next j
  next i
end sub

' Read and save the program control widget locations
sub ReadPgrmCtrlLocations
  local i, j
  for i = 0 to 3
    for j = 0 to 3
      read pctlocs(j, i)
    next j
  next i
end sub
  
' Reset the computer
' Stops the clock, clears all registers and control lines,
' and puts ram contents back to the start state.
sub ResetComputer
  local i
  clk = STOPPED
  clk_count = -1
  clk_state = CLK_STOPPED
  pcr = 0
  insr = 0
  mr = 0
  ustep = 0
  opstep = 0
  opstart = 0
  ar = 0
  alur = 0
  br = 0
  outr = 0
  bus = 0
  for i = 0 to NCONTROL_SIGS-1
    cntrl_lines(i) = 0
  next i
  for i = 1 to RSIZE
    ram(i) = rs_ram(i)
  next i
  num_trace_entries = 0
  trace_enable = 0
  DrawComputer
end sub  

' Draw the entire display
' This does not need to be redrawn as the computer runs;
' instead, only a few modules need to be redrawn and all the lights.
sub DrawComputer
  local i, x, y, y2, mn
  cls rgb(180, 180, 180)
  x = 0 : y = 0 : mn = 0
  ' left-hand module frames
  for i = 1 to NMODULES\2
    y2 = mheights(i-1)
    PlaceLights mn, x, y
    DrawModule mn, x, y, MWIDTH, y2, mnames$(i-1)
    inc y, y2
    inc mn
  next i
  ' right-hand module frames
  x = MWIDTH+BWIDTH : y = 0
  for i = 1 to NMODULES\2
    y2 = mheights(i+NMODULES\2-1)
    PlaceLights mn, x, y
    if i+NMODULES\2-1 = CNTRL then
      PlaceControlLights mn, x, y
    end if
    DrawModule mn, x, y, MWIDTH, y2, mnames$(i+NMODULES\2-1)
    inc y, y2
    inc mn
  next i
  ' bus
  x = MWIDTH : y = 0
  DrawBus
end sub

' Draw a specified module
sub DrawModule mn, x, y, w, h, n$
  local v, nl, cx1, cx2
  box x, y, w, h,, rgb(black)
  text x+10, y+3, n$, "LT", 7,, rgb(black), -1
  select case mn
    case CLOCK
      DrawClockControls mn, x, y
      v = clk
    case PC
      v = pcr
    case ADDREG
      v = mr
    case INSREG
      v = insr
    case RAMM
      v = ram(mr)
    case PROGRAM
      v = -1
      DrawProgramControls mn, x, y
      pctmoduleloc(0) = x : pctmoduleloc(1) = y
    case AREG
      v = ar
    case ALU
      v = alur
    case BREG
      v = br
    case OUT
      v = -1
      DrawDigitalDisplay
    case FLAGS
      v = flagr
      text x+72+5*LTSPAC, y+38, "C", "CT", 7,, rgb(black), -1
      text x+72+4*LTSPAC, y+38, "Z", "CT", 7,, rgb(black), -1
      text x+72-2*LTSPAC, y+38, "N", "CT", 7,, rgb(black), -1
    case CNTRL
      v = ustep
      DrawControlLights
  end select
  DrawLights mn, v
end sub

' Draw the program controls
sub DrawProgramControls mn, x, y
  rbox x+pctlocs(0,0), y+pctlocs(1,0), pctlocs(2,0), pctlocs(3,0), 5, rgb(black), rgb(white)
  text x+pctlocs(0,0)+pctlocs(2,0)\2, y+pctlocs(1,0)+7, "Load", "CT",7,, rgb(black), -1
  box x+pctlocs(0,2), y+pctlocs(1,2), pctlocs(2,2), pctlocs(3,2),, rgb(black), rgb(white)
  if len(selected_program$) > 0 then
    text x+pctlocs(0,2)+10, y+pctlocs(1,2)+pctlocs(3,2)\2, selected_program$, "LM",,, rgb(black), -1
  end if
  text x+pctlocs(0,0), y+pctlocs(1,3)+7, "Status: ", "LT", 7,, rgb(black), -1
  if len(status_msg$) > 0 then
    text x+pctlocs(0,0)+46, y+pctlocs(1,3)+4, status_msg$,,,, rgb(black), -1
  end if
end sub

' Handle mouse clicks in program controls
' (also used when there is no mouse)
sub CheckProgramControls x, y
  local px, py
  px = 0 : py = 5*MHEIGHT
  if x >= px+pctlocs(0,0) and x <= px+pctlocs(0,0)+pctlocs(2,0) then
    if y >= py+pctlocs(1,0) and y <= py+pctlocs(1,0)+pctlocs(3,0) then
      loading_mode = 1  
      DoShowPrograms 
    end if
  end if
end sub

' Place the leftmost light in each module
sub PlaceLights mn, x, y
  light_locs(0, mn) = x + light_data(0, mn)
  light_locs(1, mn) = y + light_data(1, mn)
end sub

' Place the control line lights
sub PlaceControlLights mn, x, y
  local i
  text x+0.3*MWIDTH, y+27, "Microstep", "LT", 7,, rgb(black), -1
  for i = 0 to NCONTROL_SIGS-1
    if i < NCONTROL_SIGS\2 then
      ctrl_light_locs(0, i) = 31 + x + 2*(i+1)*LTSPAC
      ctrl_light_locs(1, i) = y+80
    else
      ctrl_light_locs(0, i) = 30 + x + 2*(i-NCONTROL_SIGS\2+1)*LTSPAC
      ctrl_light_locs(1, i) = y+110
    end if
  next i
end sub

' Draw all the lights for a module
sub DrawLights mn,  v
  local lx, ly, nl, cx1, cx2, c, c2, i, b, tv
  lx = light_locs(0, mn)
  ly = light_locs(1, mn)
  nl = light_data(2, mn)
  cx1 = light_data(3, mn)
  cx2 = light_data(4, mn)
  tv = v
  if mn = CLOCK then
    if clk = 1 then
      circle lx, ly, LTRAD,,, rgb(black),  rgb(0, 0, 100)
      circle lx+LTSPAC, ly, LTRAD,,, rgb(black), rgb(0, 0, 255)
    else
      circle lx, ly, LTRAD,,, rgb(black),  rgb(0, 0, 255)
      circle lx+LTSPAC, ly, LTRAD,,, rgb(black), rgb(0, 0, 100)
    end if      
  else
    for i = 0 to nl-1
      b = tv and &h1
      if cx2 = cx1 then
        c = colors(b, cx1)
      else
        if i <= 3 then c = colors(b, cx1) else c = colors(b, cx2)
      end if
      circle lx+(nl-i)*LTSPAC, ly, LTRAD,,, rgb(black), c 
      tv = tv >> 1
    next i        
  end if
end sub  

' Set values for all lights and then call the redraw routine
sub UpdateLights
  local mn, v
  for mn = 0 to NMODULES-1
    select case mn
      case CLOCK
       v = clk
      case PC
        v = pcr
      case ADDREG
        v = mr
      case INSREG
        v = insr
      case RAMM
        v = ram(mr)
      case ASSM
        v = -1
      case AREG
        v = ar
      case ALU
        v = alur
      case BREG
        v = br
      case OUT
        v = -1
      case FLAGS
        v = flagr
      case CNTRL
        v = ustep
    end select
    DrawLights mn, v
  next mn
end sub

' Draw the various controls for the clock
sub DrawClockControls mn, x, y
  local rc = rgb(white)
  local wx, wy, ww
  if clk_state = CLK_RUNNING then
    rc = rgb(black)
  else
    rc = rgb(white)
  end if
  ' Reset button
  wx = x+clk_ctrls(0,0)
  wy = y+clk_ctrls(1,0)
  ww = clk_ctrls(2,0)
  wh = clk_ctrls(3,0)
  rbox wx, wy, ww, wh, 5, rgb(black), rgb(white)
  text wx+ww\2, wy+wh\2, "Reset", "CM", 7,, rgb(black), -1
  text wx+ww+5, wy+wh\2, "F2", "LM", 7,, rgb(black), -1
  ' Run button
  wx = x+clk_ctrls(0,1)
  wy = y+clk_ctrls(1,1)
  wr1 = clk_ctrls(2,1)
  wr2 = clk_ctrls(3,1)
  circle wx, wy, wr1,,, rgb(black)
  if clk_state <> CLK_STOPPED then
    rc = rgb(black)
  else
    rc = rgb(white)
  end if
  circle wx, wy, wr2,,, rgb(black), rc
  text wx+13, y+clk_ctrls(1,1), "Run F3", "LM", 7,, rgb(black), -1
  ' Stop button
  wx = x+clk_ctrls(0,2)
  wy = y+clk_ctrls(1,2)
  wr1 = clk_ctrls(2,2)
  wr2 = clk_ctrls(3,2)
  circle wx, wy, wr1,,, rgb(black)
  if clk_state = CLK_STOPPED then
    rc = rgb(black)
  else
    rc = rgb(white)
  end if
  circle wx, wy, wr2,,, rgb(black), rc
  text x+clk_ctrls(0,2)+13, y+clk_ctrls(1,2), "Stop F4", "LM", 7,, rgb(black), -1
  ' Step button
  wx = x+clk_ctrls(0,3)
  wy = y+clk_ctrls(1,3)
  ww = clk_ctrls(2,3)
  wh = clk_ctrls(3,3)
  rbox wx, wy, ww, wh, 5, rgb(black), rgb(white)
  text wx+ww\2, wy+wh\2, "Step", "CM", 7,, rgb(black), -1
  text wx+ww+5, wy+wh\2, "F5", "LM", 7,, rgb(black), -1
  ' Speed slider
  DrawSpeedSlider x, y
end sub

' Draw the clock speed slider
sub DrawSpeedSlider x, y
  circle x+prv_clk_slider, y+clk_ctrls(3,4), 10,,, rgb(180, 180, 180), rgb(180, 180, 180)
  line x+clk_ctrls(0,4), y+clk_ctrls(1,4), x+clk_ctrls(2,4), y+clk_ctrls(3,4), 3, rgb(black)
  circle x+clk_slider, y+clk_ctrls(3,4), 10,,, rgb(black), rgb(white)
  prv_clk_slider = clk_slider
  text x+150, y+95, "Speed", "RB", 7,, rgb(black), -1
end sub

' Draw the simulated 3-character LCD display on the Output module
sub DrawDigitalDisplay
  local x, y, rx, ry
  local rc = rgb(white)
  x = MWIDTH+BWIDTH
  y = 3*MHEIGHT
  rx = 180
  ry = 30
  circle x+rx, y+ry, 7,,, rgb(black)
  if digit_mode = dHEXRAW then
    rc = rgb(black)
  else
    rc = rgb(white)
  end if
  circle x+rx, y+ry, 4,,, rgb(black), rc
  text x+rx+13, y+ry, "Hex", "LM", 7,, rgb(black), -1
  inc ry, 18
  circle x+rx, y+ry, 7,,, rgb(black)
  if digit_mode = dUNSIGNED then
    rc = rgb(black)
  else
    rc = rgb(white)
  end if
  circle x+rx, y+ry, 4,,, rgb(black), rc
  text x+rx+13, y+ry, "Unsigned Decimal", "LM", 7,, rgb(black), -1
  inc ry, 18
  circle x+rx, y+ry, 7,,, rgb(black)
  if digit_mode = dSIGNED then
    rc = rgb(black)
  else
    rc = rgb(white)
  end if
  circle x+rx, y+ry, 4,,, rgb(black), rc
  text x+rx+13, y+ry, "Signed Decimal", "LM", 7,, rgb(black), -1
  digits_x = x+30 : digits_y = y+30
  DrawDigits
end sub

' This draws the actual value characters on the output LCD
' Using 4 simulated 7-segment chars (3 chars max plus sign)
sub DrawDigits
  local d$
  local dval
  DrawNumber digits_x, digits_y, outr, digit_mode
end sub

' Draw the control lights on the Control module
sub DrawControlLights
  local i, lx, ly, c
  ly = y + 80
  for i = 0 to NCONTROL_SIGS-1
    c = colors(cntrl_lines(i), 0)
    lx = ctrl_light_locs(0, i)
    ly = ctrl_light_locs(1, i)
    circle lx, ly, LTRAD,,, rgb(black), c
    text lx, ly-LTRAD-1, ctrlsig_names$(i), "CB", 7,, rgb(black), -1
  next i  
end sub

' Draw the Bus
sub DrawBus
  local i, x, y, b, tb, c
  x = MWIDTH
  y = 50
  box x, 0, BWIDTH, MM.VRES,, rgb(black)
  text x + BWIDTH\2, 3, "Bus", "CT", 7,, rgb(black), -1
  tb = bus
  for i = 0 to 7
    b = tb and &H1  
    c = colors(b, 0)
    circle x+30+(i-1)*LTSPAC, y, LTRAD,,, rgb(black),  c
    tb = tb >> 1
  next i
end sub

' Initialize the mouse and cursor
sub InitMouse
  on error skip 1
  controller mouse open MCHAN, MouseClick
  if MM.ERRNO <> 0 then
    hasMouse = 0
  else
    gui cursor on 1, 50, 50, rgb(green)
    settick 40, SyncCursor, 1
    hasMouse = 1
  end if
end sub

' Make cursor track the Mouse
sub SyncCursor
  local x, y
  x = Mouse(X)
  y = Mouse(Y)
  gui cursor x, y
end sub

' Handle left-clicks
sub MouseClick
  local x, y
  x = Mouse(X) : y = Mouse(Y)
  CheckClockControls x, y
  CheckOutputControls x, y
  CheckProgramControls x,y
end sub

' See if the mouse was clicked in any of the clock controls
' Note: does not yet support mouse dragging for the speed.)
sub CheckClockControls x, y
  local i, cx, cy, cw, cl, h
  for i = 0 to 4
    h = 0
    select case i
      case 0
        cx = clk_ctrls(0, i)
        cy = clk_ctrls(1, i)
        cw = clk_ctrls(2, i)
        ch = clk_ctrls(3, i)
      case 1 to 2
        cx = clk_ctrls(0, i) - clk_ctrls(2, i)\2
        cy = clk_ctrls(1, i) - clk_ctrls(2,i)\2
        cw = 2*clk_ctrls(2,i)
        ch = cw
      case 3
        cx = clk_ctrls(0, i)
        cy = clk_ctrls(1, i)
        cw = clk_ctrls(2, i)
        ch = clk_ctrls(3, i)
      case 4
        cx = clk_ctrls(0, i)
        cy = clk_ctrls(1, i) - 10
        cw = clk_ctrls(2, i) - clk_ctrls(0, i)
        ch = 20
    end select
    if x >= cx and x <= cx+cw and y >= cy and y <= cy+ch then
      select case i
        case 0
          ResetComputer
        case 1
          clk_state = CLK_RUNNING
          StartClock
        case 2
          clk_state = CLK_STOPPED
          StopClock
        case 3
          clk_state = CLK_STOPPED
          StepClock
        case 4
          clk_slider = x
          clk_rate = int(10*(300-x))+10
      end select
      DrawClockControls CLOCK, 0, 0
      exit for
    end if
  next i
end sub

' Check for Keyboard Clock speed increments
sub  ChangeClockSpeed d
  local x
  x = clk_slider
  inc x, d*CLK_SLIDER_INCREMENT
  if x < clk_ctrls(0, 4) then x = clk_ctrls(0, 4)
  if x > clk_ctrls(2, 4) then x = clk_ctrls(2, 4)
  clk_slider = x
  clk_rate = int(10*(300-x))+10
  DrawSpeedSlider 0,0
end sub

' Check for a click in the output display modes
sub CheckOutputControls x, y
  local i, mx, my, cx, cy, cw
  mx = MWIDTH+BWIDTH
  my = 3*MHEIGHT
  for i = 0 to 2
    cx = mx + digital_display_locs(0, i)
    cy = my + digital_display_locs(1, i)
    cw = digital_display_locs(2, i)
    ch = digital_display_locs(3, i)
    if x >= cx and x <= cx+cw and y >= cy and y <= cy+ch then
      select case i
        case 0
          digit_mode = dHEXRAW
        case 1
          digit_mode = dUnsigned
        case 2
          digit_mode = dSigned
      end select
      DrawDigitalDisplay
      exit for
    end if
  next i
end sub

' Clean up and exit the program
sub Quit
  if hasMouse then
    settick 0, SyncCursor
    controller mouse close
    gui cursor hide
  end if
  close #1
  cls
  end
end sub

' Cycle the clock once
sub StepClock
  clk = 1
  ClockRisingEdge
  DrawModule CLOCK
  pause 500
  clk = 0
  ClockFallingEdge
  DrawModule CLOCK
  inc clk_count
end sub

' Start the clock
sub StartClock
  settick clk_rate\2, TickClock, 2
end sub

' Stop the clock
sub StopClock
  clk_count = 0
  settick 0, 0, 2
  clk = STOPPED
  clk_state = CLOCK_STOPPED
end sub

' Run the clock at the current speed
sub TickClock
  clk = 1 - clk
  if clk = 1 then
    ClockRisingEdge
    inc clk_count
  else
    ClockFallingEdge
  end if
  DrawModule CLOCK
end sub

' On the clock rising edge, all the data transfers using the bus and
' all memory accesses are done, and the program counter is incremented or jumped.
' This is controlled by the control signal lines which are set during clock falling edges.
' The actions are handled in two batches so that all 'output to bus' controls are handled
' before any 'input from bus' controls, so as to ensure that the bus values are written before
' being read.
sub ClockRisingEdge
  local i
  opcode = insr >> 4
  if trace_enable then
    if num_trace_entries < MAX_TRACES-1 then
      tr(0, num_trace_entries) = pcr
      tr(1, num_trace_entries) = opcode
      tr(2, num_trace_entries) = ar
      tr(3, num_trace_entries) = br
      tr(4, num_trace_entries) = flagr
     inc num_trace_entries)
    end if
  end if
  for i = 0 to NCONTROL_SIGS-1
    if cntrl_lines(i) = 1 then
      select case i
        case HLT
          StopClock
        case RO
          if mr > RSIZE-1 then
            cls
            print "BUG: memory register value exceeds memory size"
            StopClock
          end if
          bus = ram(mr)
        case IO
          bus = insr and &Hf
        case CO
          bus = pcr
        case CE
          inc pcr
        case AO
          bus = ar
        case EO
          bus = alur
        case MI
          mr = bus
        case RI
          ram(mr) = bus
        case II
          insr = bus
          opcode = insr >> 4
        case JMP
          if opcode = opJMP then
            pcr = bus
          else if opcode = opJC then
            if (flagr and CARRY_BIT) <> 0 then
              pcr = bus
            end if
          else if opcode = opJZ then
            if flagr and ZERO_BIT) <> 0 then 
              pcr = bus
            end if
          else if opcode = opJN then
            if (flagr and NEG_BIT) <> 0 then
              pcr = bus
            end if
          end if
        case AI
          ar = bus
          ar_zero = 0
          ar_neg = 0
          if ar = 0 then
            ar_zero = 1
          end if
          if (ar and NEG_BIT) <> 0 then
            ar_neg = 1
          end if
          UpdateALU
        case SU
          ' handled by AI and BI
        case BI
          br = bus        
          UpdateALU
        case OI
          outr = bus
          DrawDigits
      end select
    end if
  next i            
  if cntrl_lines(FI) then
    flagr = 0
    if ar_zero then flagr = flagr or ZERO_BIT
    if ar_neg then flagr = flagr or NEG_BIT
    if alu_carry_out then flagr = flagr or CARRY_BIT
  end if
  DrawBus
  UpdateLights
end sub

' Update the ALU by computing the sum or difference of A and B Registers.
' The alu_carry_out signal goes high whenever the alu register
' overflows into bit 8. After the alu register is updated, it gets
' clipped to the lowest significant 8 bits.
' We do an explicit two's complement of the subtrahend, determine the
' carry out, and then clip the difference to 8 bits.
' Note that UpdateALU is called whenever either the A Register or the
' B Register gets new input. However, we only want to set the carry out
' signal when the B Register gets new input, because otherwise the carry out
' signal will be cleared too early and the JC conditional jump will fail.
' The zero bit used by the JZ conditional jump is set in the clock_rising_edge
' method, as is the 'negative bit' used by the JN instruction.
sub UpdateALU
  local tcb
  if cntrl_lines(SU) = 0 then
    alur = ar + br
  else
    tcb = 256 - br
    alur = ar + tcb
  end if
  if cntrl_lines(BI) > 0 then
    alu_carry_out = 0
    if alur > 255 then
      alu_carry_out = 1
    end if
  end if
  alur = alur and &B11111111
end sub

' Do all state change for a clock falling edge
' On falling edges, the microstep index is incremented. The first two
' microsteps are for fetches and the remaining ones are for opcode steps.
' All control lines are updated as needed in readiness for the next clock
' rising edge. The microstep index is reset after all steps required for
' the current opcode have been executed.
' (Be careful about making changes to this code: seemingly harmless changes
' will mess up the computer!)
' ustep: microstep index 0...4
' opcode: current opcode, will be zero (NOP) during fetch!
' opstep: step for current operation 0..n
' opaddr: line of the mcr_ops ROM currently being used to set control signals
' op_nsteps: number of steps required for current opcode to complete (includes fetch)
sub ClockFallingEdge
  trace_enable = 0
  if ustep < 2 then
    opstart = 0
    FetchStepCtrl ustep
  else
    if ustep = 2 then
      opstart = 1
      opcode = insr >> 4
      opaddr = mcr_index(0, opcode)
      op_nsteps = mcr_index(1, opcode)
      opstep = 0
    end if
    OpStepCtrl opcode, opaddr
    DrawControlLights
    if opstep < op_nsteps-1 then
      inc opstep
      inc opaddr
    end if
  end if
  inc ustep
  if ustep > 2 and ustep >= op_nsteps then
    ustep = 0 
    trace_enable = 1
    trace_address = pcr
  end if
end sub

' Set control signals for a fetch step
' The +2 offset is to skip the initial opcode and counts in each mcr_fetch line
sub FetchStepCtrl ustep
  local i
  for i = 0 to NCONTROL_SIGS-1
    cntrl_lines(i) = mcr_fetch(i+2, ustep)
  next i    
  'PrintOpValues
end sub

' Set control signals for an operation step
' The +2 offset is to skip the initial opcode and counts in each mcr_ops line
sub OpStepCtrl opcode, opaddr
  local i
  for i = 0 to NCONTROL_SIGS-1
    cntrl_lines(i) = mcr_ops(i+2, opaddr)
  next i
  'PrintOpValues
end sub

' (debug) print all control signal values
sub PrintControlSigs phase$, row
  local i
  for i = 0 to NCONTROL_SIGS-1
  next i
end sub    

' (debug) print relevant register values during op phase
sub PrintOpValues 
  local i
  if clk_count < 0 then exit sub
  if opstart = 0 then
    print #1, "Clock Rising edge: fetch step ";ustep
  else
    print #1, "Clock Rising Edge: opcode: ";opcode;" opstep: ";opstep;" opaddr: ";opaddr
  end if
  for i = 1 to NCONTROL_SIGS-1
    if cntrl_lines(i) = 1 then
      print #1, ctrlsig_names$(i);": ";
      select case i
        case HLT
          print #1, "clock halted"
        case RO
          print #1, "ram(mr): ";bin$(ram(mr), 8)
        case IO
          print #1, "insr: ";bin$(insr, 4)
        case CO
          print #1, "pcr: ";bin$(pcr, 4)
        case CE
          print #1, "pcr: ";bin$(pcr, 4)
        case AO
          print #1, "ar: ";bin$(ar, 8) 
        case EO
          print #1, "alur: ";bin$(alur, 8)
        case FI
          print #1, "flagr: ";bin$(flagr, 8)
        case SU
          print #1, "alur: ";bin$(alur, 8)
        case MI
          print #1, "adr: ";bin$(adr, 4)
        case RI
          print #1, "ram(mr): ";bin$(ram(mr), 8)
        case II
          print #1, "insr: ";bin$(insr, 8)
        case J
          print #1, "pcr: ";bin$(pcr, 4)
        case AI
          print #1, "ar: ";bin$(ar, 8)
        case BI
          print #1, "br: ";bin$(br, 8)
        case OI
          print #1, "outr: ";bin$(outr, 8)
      end select
    end if
  next i
end sub  

' (debug) print memory contents
sub PrintMemory
  local i
  print #1, "Memory Contents"
  for i = 0 to RSIZE-1
    print #1, i;": ";bin$(ram(i), 8)
  next i
end sub

' (debug) Test Numbers
sub TestNumbers
  local m, i
  local x(11) = (10, 50, 90, 130, 170, 210, 250, 290, 330, 370, 410, 450)
  local y(11) = (0, 50, 100, 150, 200, 250, 300, 350, 400, 450, 500, 550)
  local n(11) = (8, 13, 37, 233, 8, 13, 37, 233, 8, 13, 37, 233)
  m = dHexRaw
  for i = 0 to 3
    DrawNumber x(i), y(i), n(i), m
  next i
  m = dUnsigned
  for i = 4 to 7
    DrawNumber x(i), y(i), n(i), m
  next i
  m = dSigned
  for i = 8 to 11
    DrawNumber x(i), y(i), n(i), m
  next i
end sub

' Draw a number that needs up to 3 digits (decimal or hexadecimal),
' with a leading minus sign if needed and blanking where no digits
' are needed.
sub DrawNumber x, y, num, digit_mode
  local dnum, anum, mxdig, low, med, high, sign
  local dx, i
  local dig(4)
  select case digit_mode
    case dHexRaw
      dnum = num
      mxdig = 2
      dig(0) = dnum mod 16
      dig(1) = dnum >> 4
      dig(2) = BLANK
      dig(3) = BLANK
    case dUnsigned
      if num >= 0 then dnum = num else dnum = 256 + num
      mxdig = 3
      dig(0) = dnum mod 10
      dig(1) = ((dnum - low) mod 100)\10
      dig(2) = (dnum - 10*med - low)\100
      if dig(2) = 0 then dig(2) = BLANK
      if dig(2) = BLANK and dig(1) = 0 then dig(1) = BLANK
      dig(3) = BLANK
      mxdig = 1
      if med > 0 then mxdig = 2
      if high > 0 then mxdig = 3
      sign = 0
    case dSigned
      if num <= 127 then 
        dnum = num
      else 
        dnum = 256 - num
        sign = 1
      end if
      mxdig = 1
      if med > 0 then mxdig = 2
      if high > 0 then mxdig = 3
      anum = abs(dnum)
      dig(0) = anum mod 10
      dig(1) = ((anum - low) mod 100)\10
      dig(2) = (anum - 10*med - low)\100 
      if dig(2) = 0 then dig(2) = BLANK
      if dig(2) = BLANK and dig(1) = 0 then dig(1) = BLANK
      if sign then dig(3) = MINUS else dig(3) = BLANK
  end select
  for i = 3 to 0 step -1
    dx = x + (3-i)*SDISW
    DrawSevenSegNumber dx, y, dig(i)
  next i
end sub

' Draw a number with a simulated 7-segment display
' at the designated coordinates with the designated value shown.
' num: 0..9:   the decimal or hexadecimal value
' num: 10..15: the hexadecimal value
' num: 16: minus sign
' num: 17: blank
sub DrawSevenSegNumber x, y, num
  local lx, ly, digit, seg, active, j, k, c, dindex
  local xv(7), yv(7), digs(4)
  
  if num < 0 or num > 17 then exit sub
  c = RGB(RED)
  box x, y, SDISW, SDISH,, RGB(BLACK), RGB(BLACK)
    
  for seg = 0 to NSEGS-1
    active = SegActivations(num, seg)
    if active then
      if seg = XBAR then k = 6 else k = 4
      for j = 0 to k-1
        xv(j) = x + SegCoords(seg, j, 0)
        yv(j) = y + SegCoords(seg, j, 1)
      next j      
      xv(k) = xv(0) : yv(k) = yv(0)
      polygon k+1, xv(), yv(), c, c
    end if
  next seg  
end sub

' read the coordinates for the 7-segment numeric displays
sub ReadSegCoords
  local seg, i, j, k
  for seg = 0 to NSEGS-1
    k = 4
    if seg = XBAR then k = 6
    for i = 0 to k-1
      for j = 0 to 1
        read SegCoords(seg, i, j)
      next j
    next i
  next seg
end sub

' Read the segment activations for the seven-segment display
' for the digits 0..9
sub ReadSegActivations
  local seg, i

  for i = 0 to NFIGS-1
    for seg = 0 to NSEGS-1
      read SegActivations(i, seg)
    next seg
  next i
end sub

' Display status message in program module
sub ShowStatus m$
  local x, y
  x = pctlocs(0, 0) + 40
  y = 400 + pctlocs(1, 3)
  text x, y, m$
end sub

' Clear memory
sub ClearMemory
  local i
  for i = 1 to RSIZE
    ram(i) = 0
    rs_ram(i) = 0
  next i
end sub

' Load a progam (.bin) file
sub DoShowPrograms
  local x, y, i
  local path$
  pmode = 2
  pselect = 1
  DrawProgramlist
end sub

' Load the Chosen Program
sub ChooseProgram
  if nprograms > 0 and pselect > 0 then
    ClearMemory
    x = 0 : y = 5*MHEIGHT
    path$ = "Programs/" + program_names$(pselect)
    text x+pctlocs(0,2)+2, y+pctlocs(1,2)+7, "Programs/" + path$, "LT", 7,, rgb(black), -1
    loading_mode = 0
    LoadMemory path$
    DrawComputer
  end if
end sub

' Print a list of available programs (Programs/*.asm)
sub DrawProgramList
  local f$
  local x, y, i
  nprograms = 0
  f$ = dir$("Programs/*.bin", FILE)
  do while f$ <> ""
    inc nprograms
    program_names$(nprograms) = f$
    f$ = dir$()
  loop
  box 50, 50, 350, 450,, rgb(white), rgb(black)
  x = 55 : y = 60
  for i = 1 to nprograms
    if i = pselect then
      text x, y, program_names$(i), "LT", 1,, rgb(white), rgb(blue)
    else
      text x, y, program_names$(i), "LT", 1,, rgb(white), -1
    end if
    inc y, 15
  next i
end sub

' Load RAM from the selected .bin file
' The .bin files all have 16 lines corresponding to the 16 ram locations
' Each line has 8 binary values, e.g. 00011110
sub LoadMemory path$
  local buf$, num, addr, bit
  on error skip 1
  open path$ for input as #2
  if mm.errno <> 0 then
    ShowStatus "Error reading file '" + path$ + "'"
    exit sub
  end if
  for addr = 0 to RSIZE-1
    line input #2, buf$
    num = 0
    for bit = 1 to 8
      k = val(MID$(buf$, bit, 1))
      num = 2*num + k
    next bit  
    ram(addr) = num
    rs_ram(addr) = num
  next addr
  close #2
  PrintMemory
  ResetComputer
  selected_program$ = path$
  status_msg$ =  "Successful Load"
  DrawProgramControls 5, 445
end sub

' Process keystrokes
sub HandleKeyboardEvents
  local z$
  local cmd
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case UP
        if loading_mode then
          if pselect > 1 then
            inc pselect, -1
            DrawProgramList
          end if
        else
          if digit_mode > dHEXRAW then
            inc digit_mode, -1
            DrawDigitalDisplay
          end if          
        end if
      case DOWN
        if loading_mode then
          if pselect < nprograms then
            inc pselect
            DrawProgramList
          end if
        else
          if digit_mode < dSIGNED then
            inc digit_mode
            DrawDigitalDisplay
          end if
        end if
      case LEFT
        if loading_mode = 0 then ChangeClockSpeed -1
      case RIGHT
        if loading_mode = 0 then ChangeClockSpeed 1
      case ENTER
        if loading_mode = 1 then ChooseProgram
      case F1
        ShowHelp
      case F2
        if loading_mode = 0 then CheckClockControls 30, 30        
      case F3
        if loading_mode = 0 then CheckClockControls 35, 55
      case F4
        if loading_mode = 0 then CheckClockControls 35, 72
      case F5
        if loading_mode = 0 then CheckClockControls 160, 30
      case F8
        DrawExecutionTrace
      case F12
        loading_mode = 1
        DoShowPrograms
      case ESC
        Quit
    end select
  loop
end sub

' Show a command summary
sub ShowHelp
  local y, z$
  if hasMouse then gui cursor hide
  box 100, 100, 600, 300,, rgb(white), rgb(black)
  text 410, 110, "SAP1 Command Summary", "CT", 4,, rgb(green)
  y = 160
  print @(110, y);"F1   Repeat This Help Message"
  inc y, 15
  print @(110, y);"F2   Reset the Computer"
  inc y, 15
  print @(110, y);"F3   Start the Clock (Run)"
  inc y, 15
  print @(110, y);"F4   Stop the Clock"
  inc y, 15
  print @(110, y);"F5   Step the Clock 1 Cycle"
  inc y, 15
  print @(110, y);"F8   Show an execution trace"
  inc y, 15
  print @(110, y);"F12  Load a new binary program"
  inc y, 15
  print @(110, y);"UP   (load program mode) move up the list of programs"
  inc y, 15
  print @(110, y);"     (run mode) choose digital display type"
  inc y, 15
  print @(110, y);"DOWN (load program mode) move down the list of programs"
  inc y, 15
  print @(110, y);"     (run mode) choose digital display type"
  inc y, 15
  print @(110, y);"ENTER: choose the program to load"
  inc y, 15
  print @(110, y);"Esc  Quit the program:"
  text 410, 390, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  if hasMouse then gui cursor show
  DrawComputer
end sub

' Draw the execution trace
sub DrawExecutionTrace
  local i, row, col, m$, p$, pl
  cls
  gui cursor hide
  row = 1
  col = 1
  text 10, 0, "pcr op   areg    breg   flags"
  for i = 0 to num_trace_entries-1
    if i = MAX_TRACE_ROWS then
      col = 2
      row = 1
      text 410, 0, "pcr op   areg    breg   flags"
    end if
    p$ = op_names$(tr(1,i))
    pl = len(p$)
    if pl < 3 then cat p$, space$(3-pl)
    m$ = BIN$(tr(0,i),4) + "  " + p$ + "  " + BIN$(tr(2,i),8) + "  " + BIN$(tr(3,i),8)
    m$ = m$ + "  " + BIN$(tr(4,i),8)
    text 10 + (col-1)*400, 10 + row*15, m$, "LT", 7
    inc row
  next i
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  if hasMouse then gui cursor show
  DrawComputer
end sub

' Only data statements beyond this point!

' module heights
data 100, 100, 100, 100, 100, 100
data 100, 100, 100, 100, 60, 140

' module names
data "Clock", "Program Counter", "Address Register", "Instruction Register", "Memory", "Program"
data "Register A", "Sum Register", "Register B", "Output", "Flags Register", "Control"

' module light info:
' x, y light start location as fraction of module width, height
' number of lights
' light color indices
data 0.75, 0.3,  2, 3, 3
data  0.1, 0.5,  4, 0, 0
data  0.1, 0.5,  4, 0, 0
data  0.1, 0.5,  8, 3, 2
data  0.1, 0.5,  8, 0, 0
data  0.0, 0.0,  0, 0, 0

data  0.1, 0.5,  8, 0, 0
data  0.1, 0.5,  8, 0, 0
data  0.1, 0.5,  8, 0, 0
data  0.0, 0.0,  0, 0, 0
data  0.1, 0.5,  8, 3, 3 
data  0.3, 0.3,  5, 2, 2

' control signal names
data "HLT", "RO", "IO", "CO", "CE", "AO", "EO", "FI", "SU"
data "MI", "RI", "II", "J", "AI", "BI", "OI"

' color values for 'OFF' and 'ON' lights
data rgb(100, 0, 0), rgb(255, 0, 0)
data rgb(50, 50, 0), rgb(255, 255, 0)
data rgb(0, 75, 0), rgb(0, 255, 0)
data rgb(0, 0, 0), rgb(0, 0, 255)

data CHECK_DATA_ROOT+1

' Clock Controls Locations
data 20, 20, 40, 20
data 30, 52, 7, 4
data 30, 70, 7, 4
data 150, 20, 40, 20
data 100, 70, 300, 70

data CHECK_DATA_ROOT+2

' Microcode ROM contents
' First two numbers of each line are opcode and number of microsteps.
' the rest of the 16 numbers are the control lines as labeled.
' '99' in the opcode field for the fetch are don't cares (X's)
' Microsteps 0 and 1 are the fetch, which precedes every instruction sequence.
' There are currently a maximum of 5 microsteps per instruction inluding the fetch.
' The order of the control signals is sorted so that all the signals that cause
' a value to be asserted onto the bus happen before the signals that cause the
' bus value to be read. Signals that don't affect the bus can be located anywhere.
' Note that J, JC, JZ, and JN all use the same control line. The conditions for JC, JZ, and JN
' are handled in code without needing another control line.
'             H
'             L  R  I  C  C  A  E  F  S  M  R  I     A  B  O'
'             T  O  O  O  E  O  O  I  U  I  I  I  J  I  I  I'
data 99, 0,   0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0' fetch 1
data 99, 1,   0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0' fetch 2

data 00, 3,   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0' NOP 1
data 01, 4,   0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0' LDA 1
data 01, 4,   0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0' LDA 2
data 02, 5,   0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0' ADD 1
data 02, 5,   0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0' ADD 2
data 02, 5,   0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0' ADD 3
data 03, 5,   0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0' SUB 1
data 03, 5,   0, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0' SUB 2
data 03, 5,   0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0' SUB 3
data 04, 4,   0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0' STA 1
data 04, 4,   0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0' STA 2
data 05, 3,   0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0' LDI 1
data 06, 3,   0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0' J
data 07, 3,   0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0' JC
data 08, 3,   0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0' JZ
data 09, 0,   0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0' JN
data 10, 0,   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0' NOP2
data 11, 0,   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0' NOP2
data 12, 0,   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0' NOP2
data 13, 0,   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0' NOP2
data 14, 2,   0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1' OUT
data 15, 2,   1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0' HLT

' Contents of the mcr_index array used to index the above mcr_fetch and mcr_ops arrays.
' Start row and row count in microstep ROM for each instruction
' Count includes the 2-cycle fetch.
' Dummy NOP ops are used at adresses 9-14 as stand-ins for operations not
' defined yet.
data 0,  3   ' NOP
data 1,  4   ' LDA
data 3,  5   ' ADD
data 6,  5   ' SUB
data 9,  4   ' STA
data 11, 3   ' LDI
data 12, 3   ' J
data 13, 3   ' JC
data 14, 3   ' JZ
data 15, 3   ' JN
data 0,  3   ' NOP, dummy to fill slot
data 0,  3   ' NOP, dummy to fill slot
data 0,  3   ' NOP, dummy to fill slot
data 0,  3   ' NOP, dummy to fill slot
data 20, 3   ' OUT
data 21, 3   ' HLT

' Opcode Mnemonics (for all 16 slots)
data "NOP", "LDA", "ADD", "SUB", "STA", "LDI", "J", "JC", "JZ"
data "JN", "NOP", "NOP", "NOP", "NOP", "OUT", "HLT"

' Opcode values for all 16 slots
data &B0000, &B0001, &B0010, &B0011, &B0100, &B0101, &B0110, &B0111
data &B1000, &B1001, &B0000, &B0000, &B0000, &B0000, &B1110, &B1111

data CHECK_DATA_ROOT+3

' Seven-segment display segment polygons (relative to bounding box)
data            SS,                SS,      SDISW-SS,         SS
data SDISW-SS-SEGW,           SS+SEGW,       SS+SEGW,    SS+SEGW
data            SS,                SS,            SS,    SDISH\2
data       SS+SEGW,   SS+SDISH\2-SEGW,       SS+SEGW,    SS+SEGW
data SDISW-SS,                     SS,      SDISW-SS, SS+SDISH\2
data SDISW-SS-SEGW,   SS+SDISH\2-SEGW, SDISW-SS-SEGW,    SS+SEGW

data       SS+SEGW, SS+SDISH\2-SEGW\2,            SS, SS+SDISH\2
data       SS+SEGW, SS+SDISH\2+SEGW\2
data SDISW-SS-SEGW, SS+SDISH\2+SEGW\2,      SDISW-SS, SS+SDISH\2
data SDISW-SS-SEGW, SS+SDISH\2-SEGW\2

data            SS,        SS+SDISH\2,            SS,        SDISH-SS
data       SS+SEGW,  SS+SDISH-SS-SEGW,       SS+SEGW, SS+SDISH\2+SEGW
data      SDISW-SS,        SS+SDISH\2,      SDISW-SS,        SDISH-SS
data SDISW-SS-SEGW,     SDISH-SS-SEGW, SDISW-SS-SEGW, SS+SDISH\2+SEGW
data            SS,          SDISH-SS,      SDISW-SS,        SDISH-SS
data SDISW-SS-SEGW,     SDISH-SS-SEGW,       SS+SEGW,   SDISH-SS-SEGW

' Seven-segment activation list for numbers 0 through 9 and minus sign and blank
data 1, 1, 1, 0, 1, 1, 1 '0
data 0, 0, 1, 0, 0, 1, 0 '1
data 1, 0, 1, 1, 1, 0, 1 '2
data 1, 0, 1, 1, 0, 1, 1 '3
data 0, 1, 1, 1, 0, 1, 0 '4
data 1, 1, 0, 1, 0, 1, 1 '5
data 1, 1, 0, 1, 1, 1, 1 '6
data 1, 0, 1, 0, 0, 1, 0 '7
data 1, 1, 1, 1, 1, 1, 1 '8
data 1, 1, 1, 1, 0, 1, 1 '9
data 1, 1, 1, 1, 1, 1, 0 'A
data 0, 1, 0, 1, 1, 1, 1 'B
data 1, 1, 0, 0, 1, 0, 1 'C
data 0, 0, 1, 1, 1, 1, 1 'D
data 1, 1, 0, 1, 1, 0, 1 'E
data 1, 1, 0, 1, 1, 0, 0 'F
data 0, 0, 0, 1, 0, 0, 0 '-
data 0, 0, 0, 0, 0, 0, 0 '(blank)

data CHECK_DATA_ROOT + 4


' Digital Display locs
' 0,0 - 3,0 : x, y, w, h for mouse target #1
' 0,1 - 3,1 : x, y, w, h for mouse target #2
' 0,2 - 3,2 : x, y, w, h for mouse target #3
data 180, 30, 14, 14
data 180, 48, 14, 14
data 180, 66, 14, 14

data CHECK_DATA_ROOT + 5

' Program control widget locations
data 10, 20, 40, 20
data 10, 50, 90, 20
data 60, 20, 250, 20
data 30, 62, 250, 20

data CHECK_DATA_ROOT + 6

